#!/usr/bin/env python3
import json, subprocess, sys
from pathlib import Path

def run(cmd):
    p = subprocess.run(cmd, capture_output=True, text=True)
    return p.returncode, p.stdout, p.stderr

def load_json(p: Path):
    return json.loads(p.read_text(encoding='utf-8'))

def main():
    root = Path.cwd()
    eng  = root/'engine_a5_surface_dual.py'
    man_off = root/'config'/'manifest_off.json'
    man_on  = root/'config'/'manifest_on.json'
    diagA   = root/'config'/'diagnostics_A.json'
    diagB   = root/'config'/'diagnostics_B.json'

    outA = root/'outputs'/'A'
    outB = root/'outputs'/'B'
    outF = root/'outputs'/'final'
    outA.mkdir(parents=True, exist_ok=True)
    outB.mkdir(parents=True, exist_ok=True)
    outF.mkdir(parents=True, exist_ok=True)

    # --- Option A ---
    cmdA = [sys.executable, str(eng),
            '--manifest_off', str(man_off),
            '--manifest_on',  str(man_on),
            '--diag',         str(diagA),
            '--out',          str(outA)]
    rcA, soA, seA = run(cmdA)
    (root/'logs'/'engine_stdout_A.txt').write_text(soA)
    (root/'logs'/'engine_stderr_A.txt').write_text(seA)
    auditA = outA/'audits'/'surface_neutrality.json'
    PASS_A = False
    deltaA = None
    if auditA.exists():
        ja = load_json(auditA)
        PASS_A = bool(ja.get('PASS', False))
        deltaA = float(ja.get('delta_c_rel', 1e9))

    # If A passed, finalize
    if PASS_A:
        # copy audit to final
        (outF/'surface_neutrality.json').write_text(auditA.read_text())
        summary = {"winner":"A","delta_c_rel":deltaA,"PASS":True,
                   "audit_path": str((outF/'surface_neutrality.json').as_posix())}
        (outF/'A5_dual_summary.json').write_text(json.dumps(summary, indent=2))
        print("A5 DUAL SUMMARY:", json.dumps(summary))
        return

    # --- Option B ---
    cmdB = [sys.executable, str(eng),
            '--manifest_off', str(man_off),
            '--manifest_on',  str(man_on),
            '--diag',         str(diagB),
            '--out',          str(outB)]
    rcB, soB, seB = run(cmdB)
    (root/'logs'/'engine_stdout_B.txt').write_text(soB)
    (root/'logs'/'engine_stderr_B.txt').write_text(seB)
    auditB = outB/'audits'/'surface_neutrality.json'
    PASS_B = False
    deltaB = None
    if auditB.exists():
        jb = load_json(auditB)
        PASS_B = bool(jb.get('PASS', False))
        deltaB = float(jb.get('delta_c_rel', 1e9))

    # Choose winner / final report
    if PASS_B:
        (outF/'surface_neutrality.json').write_text(auditB.read_text())
        summary = {"winner":"B","delta_c_rel":deltaB,"PASS":True,
                   "audit_path": str((outF/'surface_neutrality.json').as_posix())}
    else:
        # both failed: pick smaller delta for triage
        winner = "A" if (deltaA or 1e9) <= (deltaB or 1e9) else "B"
        chosen = auditA if winner=="A" else auditB
        (outF/'surface_neutrality.json').write_text(chosen.read_text() if chosen.exists() else json.dumps({"error":"no audit"}, indent=2))
        summary = {"winner":winner,"delta_c_rel": min(deltaA or 1e9, deltaB or 1e9),
                   "PASS":False,
                   "audit_path": str((outF/'surface_neutrality.json').as_posix())}
    (outF/'A5_dual_summary.json').write_text(json.dumps(summary, indent=2))
    print("A5 DUAL SUMMARY:", json.dumps(summary))

if __name__ == "__main__":
    main()